'use strict';

(async function(){
  const $ = (q)=>document.querySelector(q);

  // Toolbar
  const rName = $('#rName'), weekStart = $('#weekStart'), prev = $('#prev'), next = $('#next');
  const deptChecks = $('#deptChecks'), note = $('#note');
  const validateBtn = $('#validateBtn'), saveDraft = $('#saveDraft'), publishBtn = $('#publish'), autoBtn = $('#autoPlan'), helpBtn=$('#helpBtn');
  const clearWeekBtn = $('#clearWeek');

  // Board
  const board = $('#board'), rulerWeek = $('#rulerWeek'), daysHead = $('#daysHead');
  const rowsEl = $('#rows'), overlay = $('#drawOverlay'), peopleEl = $('#people');

  // Drawer
  const drawer = $('#drawer'), drawerClose = $('#drawerClose');
  const fDept = $('#fDept'), fDate = $('#fDate'), fStart = $('#fStart'), fEnd = $('#fEnd'), fTitle = $('#fTitle');
  const fBreak = $('#fBreak'), autoBreak = $('#autoBreak');
  const upSearch = $('#upSearch'), upList = $('#upList'), upSelected = $('#upSelected'), upClear = $('#upClear'), upAll = $('#upAll');
  const drawerNote = $('#drawerNote'), deleteShiftBtn = $('#deleteShift');

  // Conflicts pane
  const confPane = $('#confPane'), confList = $('#confList'), confCount = $('#confCount');

  // Auto planner
  const autoModal = $('#autoModal'), autoClose = $('#autoClose'), autoMatrix = $('#autoMatrix');
  const genConcept = $('#genConcept'), applyAppend = $('#applyConceptAppend'), applyReplace = $('#applyConceptReplace');
  const autoPreview = $('#autoPreview'), autoNote = $('#autoNote');

  // State
  let departments = [], users = [];
  let selectedDeptIds = new Set();
  let weekMonday = getMonday(new Date());
  let lanes = []; // [{userId, name}]
  let palette = new Map(); // deptId -> c1..c8
  let shifts = []; // [{department_id,title,start_at,end_at,break_minutes,user_ids[]}]
  let concept = null;
  let selectedUserIds = new Set();
  let editingIndex = -1;
  let rosterId = 0;

  // Utils
  function getMonday(d){ const x=new Date(d.getFullYear(), d.getMonth(), d.getDate()); const n=(x.getDay()||7)-1; x.setDate(x.getDate()-n); x.setHours(0,0,0,0); return x; }
  function ymd(d){ const y=d.getFullYear(), m=String(d.getMonth()+1).padStart(2,'0'), dd=String(d.getDate()).padStart(2,'0'); return `${y}-${m}-${dd}`; }
  function addDays(d,n){ return new Date(d.getFullYear(), d.getMonth(), d.getDate()+n); }
  function parseLocal(dateStr,timeStr){ return `${dateStr} ${timeStr}:00`; }
  function escapeHtml(s){ return (s||'').replace(/[&<>"]/g, c=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;' }[c])); }
  function show(el,msg,ok=false){ el.hidden=false; el.textContent=msg; el.style.background= ok?'#ecfdf5':'#eef2ff'; el.style.borderColor= ok?'#a7f3d0':'#c7d2fe'; }
  function hide(el){ el.hidden=true; }

  // Default pauze
  function defaultBreak(date, start, end){
    const a = new Date(`${date}T${start}:00`), b=new Date(`${date}T${end}:00`);
    const mins = Math.max(0, Math.round((b-a)/60000));
    if (mins < 360) return 0; if (mins <= 480) return 30; return 45;
  }

  // Load meta
  async function loadDepartments(){
    const d = await fetch('/api/departments/list.php').then(r=>r.json());
    departments = d.ok ? d.items : [];
    selectedDeptIds = new Set(departments.slice(0,2).map(x=>Number(x.id)));
    let i=1; deptChecks.innerHTML='';
    departments.forEach(dep=>{
      const lab = document.createElement('label'); lab.className='check';
      lab.innerHTML = `<input type="checkbox" value="${dep.id}" ${selectedDeptIds.has(Number(dep.id))?'checked':''}><span>${escapeHtml(dep.name)}</span>`;
      const cb = lab.querySelector('input');
      cb.addEventListener('change', async ()=>{
        if(cb.checked) selectedDeptIds.add(Number(cb.value)); else selectedDeptIds.delete(Number(cb.value));
        await loadUsers(); buildLanes(); renderBoard();
        fDept.innerHTML = departments.map(d=>`<option value="${d.id}">${escapeHtml(d.name)}</option>`).join('');
      });
      deptChecks.appendChild(lab);
      palette.set(Number(dep.id), `c${((i-1)%8)+1}`); i++;
    });
    fDept.innerHTML = departments.map(d=>`<option value="${d.id}">${escapeHtml(d.name)}</option>`).join('');
  }
  async function loadUsers(){
    const u = await fetch('/api/users/list.php', { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({}) }).then(r=>r.json());
    users = u.ok ? u.items : [];
  }
  function buildLanes(){
    lanes = users.map(u=>({ userId: u.id, name: u.name }));
    peopleEl.innerHTML = lanes.map(l=>`<div class="person" data-user="${l.userId}"><span>${escapeHtml(l.name)}</span></div>`).join('');
  }

  // Header + ruler (✅ per dag)
  function renderRuler(){
    rulerWeek.innerHTML = '';
    for (let i=0;i<7;i++){
      const day = document.createElement('div');
      day.className = 'rulerDay';
      for (let k=0;k<=16;k++){
        const t = 6 + k; // 06..22
        const tick = document.createElement('div');
        tick.className = 'rulerTick';
        tick.style.left = `${(k/16)*100}%`;
        tick.textContent = (t<10?'0':'')+t+':00';
        day.appendChild(tick);
      }
      rulerWeek.appendChild(day);
    }
  }
  function renderHead(){
    daysHead.innerHTML = '';
    for(let i=0;i<7;i++){
      const d = addDays(weekMonday,i);
      const el = document.createElement('div'); el.className='dayHead';
      el.textContent = d.toLocaleDateString('nl-NL',{weekday:'long', day:'2-digit', month:'2-digit'});
      daysHead.appendChild(el);
    }
  }

  // Board
  function renderBoard(){
    renderHead(); renderRuler();
    rowsEl.innerHTML = '';
    lanes.forEach(l=>{
      // label
      const label = document.createElement('div'); label.className='rowLabel'; label.textContent = l.name;
      rowsEl.appendChild(label);
      // 7 aparte dagen
      const dayWrap = document.createElement('div'); dayWrap.className='rowDays';
      for (let i=0;i<7;i++){
        const dateStr = ymd(addDays(weekMonday, i));
        const lane = document.createElement('div'); lane.className='rowLane'; lane.dataset.user = l.userId; lane.dataset.date = dateStr;
        const grid = document.createElement('div'); grid.className='hourGrid';
        const band = document.createElement('div'); band.className='band';
        lane.appendChild(grid); lane.appendChild(band);
        dayWrap.appendChild(lane);
      }
      rowsEl.appendChild(dayWrap);
    });
    drawShifts();
  }
  function timeToX(dateStr, timeStr){
    const start = new Date(`${dateStr}T06:00:00`);
    const end   = new Date(`${dateStr}T22:00:00`);
    let t = new Date(`${dateStr}T${timeStr}:00`);
    if (t < start) t = start; if (t > end) t = end;
    const span = end - start || 1;
    return ((t - start)/span) * 100;
  }
  function laneFor(userId, dateStr){
    return rowsEl.querySelector(`.rowLane[data-user="${userId}"][data-date="${dateStr}"]`);
  }

  // Tekenen/plaatsen
  function drawShifts(conflicts=[]){
    // markering sets
    const markOverlap = new Set(conflicts.filter(c=>c.type==='overlap' || c.type==='business_hours')
      .map(c => `${c.type}:${c.user_id||''}:${c.date}:${c.start_at||''}:${c.end_at||''}`));
    const markMulti = new Set(conflicts.filter(c=>c.type==='multi')
      .map(c => `multi:${c.user_id}:${c.date}`));

    rowsEl.querySelectorAll('.band').forEach(b=> b.innerHTML='');

    shifts.forEach((s, idx)=>{
      const date = s.start_at.slice(0,10);
      const start = s.start_at.slice(11,16);
      const end   = s.end_at.slice(11,16);
      const left = timeToX(date, start);
      const right = timeToX(date, end);
      const width = Math.max(2, right-left);
      const cls = palette.get(Number(s.department_id)) || 'c1';
      const depName = departments.find(d=>Number(d.id)===Number(s.department_id))?.name || '';
      const peopleCount = (s.user_ids?.length || 0);

      (s.user_ids||[]).forEach(uid=>{
        const lane = laneFor(Number(uid), date);
        if (!lane) return; // uit beeld
        const band = lane.querySelector('.band');
        const bar = document.createElement('div'); bar.className='bar '+cls; bar.style.left=left+'%'; bar.style.width=width+'%';
        bar.title = `${start}–${end} • ${depName}`;
        bar.innerHTML = `${escapeHtml(start)}–${escapeHtml(end)} • ${escapeHtml(depName)} <span class="count">${peopleCount}</span>`;
        if (markOverlap.has(`overlap:${uid}:${date}:${start}:${end}`) || markOverlap.has(`business_hours::${date}:${start}:${end}`) || markMulti.has(`multi:${uid}:${date}`)) {
          bar.classList.add('err');
        }
        bar.addEventListener('click', ()=> openDrawer(idx));
        band.appendChild(bar);
      });
    });
  }

  // Drag per dag
  let drag=null;
  rowsEl.addEventListener('mousedown', (e)=>{
    const lane = e.target.closest('.rowLane'); if (!lane) return;
    const date = lane.dataset.date;
    const laneRect = lane.getBoundingClientRect(), boardRect=board.getBoundingClientRect();
    const x = (e.clientX - laneRect.left) / laneRect.width;
    drag = { laneUser:Number(lane.dataset.user), date, startX:x, endX:x, laneRect, boardRect };
    overlay.hidden=false; positionOverlay(laneRect, boardRect, x, x); e.preventDefault();
  });
  window.addEventListener('mousemove', (e)=>{
    if(!drag) return;
    const lane = laneFor(drag.laneUser, drag.date); if (!lane) return;
    const laneRect = lane.getBoundingClientRect(), boardRect=board.getBoundingClientRect();
    const x = Math.max(0, Math.min(1, (e.clientX - laneRect.left) / laneRect.width));
    drag.endX=x; positionOverlay(laneRect,boardRect,drag.startX,drag.endX);
  });
  window.addEventListener('mouseup', ()=>{
    if(!drag) return;
    const { laneUser, date, startX, endX } = drag;
    overlay.hidden=true; drag=null;
    const a=Math.min(startX,endX), b=Math.max(startX,endX); if (Math.abs(b-a)<0.02) return;
    const start=xToTime(date,a), end=xToTime(date,b);
    openDrawer(null, { date, start, end, userIds:[laneUser] });
  });
  rowsEl.addEventListener('dblclick', (e)=>{
    const lane = e.target.closest('.rowLane'); if (!lane) return;
    const date = lane.dataset.date;
    const rect = lane.getBoundingClientRect();
    const x = Math.max(0, Math.min(1, (e.clientX - rect.left) / rect.width));
    const start = xToTime(date, x); const end = addMinutesStr(start, 240);
    openDrawer(null, { date, start, end, userIds:[Number(lane.dataset.user)] });
  });
  function positionOverlay(laneRect, boardRect, a,b){
    const left = laneRect.left - boardRect.left + laneRect.width * Math.min(a,b);
    const width= laneRect.width * Math.abs(b-a);
    overlay.style.left=left+'px'; overlay.style.top=(laneRect.top - boardRect.top)+'px';
    overlay.style.width=width+'px'; overlay.style.height=laneRect.height+'px';
  }
  function xToTime(date, x){
    const start = new Date(`${date}T06:00:00`), end = new Date(`${date}T22:00:00`);
    const t = new Date(start.getTime() + (end-start)*x);
    const hh=String(t.getHours()).padStart(2,'0'); const mm=String(Math.round(t.getMinutes()/15)*15).padStart(2,'0');
    return `${hh}:${mm}`;
  }
  function addMinutesStr(hhmm, mins){
    const [h,m]=hhmm.split(':').map(n=>parseInt(n,10)); const d=new Date(2000,0,1,h,m,0); d.setMinutes(d.getMinutes()+mins);
    return `${String(d.getHours()).padStart(2,'0')}:${String(d.getMinutes()).padStart(2,'0')}`;
  }

  // Drawer + Picker
  function syncChips(){ upSelected.innerHTML=''; Array.from(selectedUserIds).forEach(id=>{ const u=users.find(x=>Number(x.id)===Number(id)); if(!u) return; const chip=document.createElement('span'); chip.className='chip'; chip.innerHTML=`${escapeHtml(u.name)} <button class="x" type="button">×</button>`; chip.querySelector('.x').addEventListener('click',()=>{ selectedUserIds.delete(id); syncChips(); renderUserList(upSearch.value);}); upSelected.appendChild(chip); }); }
  function renderUserList(q=''){ const ql=(q||'').toLowerCase(); upList.innerHTML=''; users.filter(u=>!q|| (u.name||'').toLowerCase().includes(ql) || (u.email||'').toLowerCase().includes(ql)).forEach(u=>{ const row=document.createElement('div'); row.className='up-item'; const checked=selectedUserIds.has(Number(u.id))?'checked':''; row.innerHTML=`<span>${escapeHtml(u.name)}</span><input type="checkbox" ${checked}>`; const cb=row.querySelector('input'); cb.addEventListener('change',()=>{ if(cb.checked) selectedUserIds.add(Number(u.id)); else selectedUserIds.delete(Number(u.id)); syncChips(); }); upList.appendChild(row); }); }
  upSearch.addEventListener('input', ()=> renderUserList(upSearch.value));
  upClear.addEventListener('click', ()=>{ selectedUserIds.clear(); syncChips(); renderUserList(upSearch.value); });
  upAll.addEventListener('click', ()=>{ selectedUserIds = new Set(users.map(u=>Number(u.id))); syncChips(); renderUserList(upSearch.value); });
  peopleEl.addEventListener('click', (e)=>{ if (drawer.getAttribute('aria-hidden') === 'true') return; const card = e.target.closest('.person'); if (!card) return; const uid=Number(card.dataset.user); if (selectedUserIds.has(uid)) selectedUserIds.delete(uid); else selectedUserIds.add(uid); syncChips(); renderUserList(upSearch.value); });

  function openDrawer(index, preset){
    drawer.setAttribute('aria-hidden','false'); drawerNote.hidden=true; deleteShiftBtn.hidden = !(index>=0); editingIndex = (index>=0 ? index : -1);
    const ref = (index>=0 ? shifts[index] : null);
    const date = preset?.date || (ref ? ref.start_at.slice(0,10) : ymd(weekMonday));
    const start = preset?.start || (ref ? ref.start_at.slice(11,16) : '09:00');
    const end = preset?.end || (ref ? ref.end_at.slice(11,16) : '17:00');

    fDate.value=date; fStart.value=start; fEnd.value=end; fTitle.value = ref?.title || '';
    fDept.innerHTML = departments.map(d=>`<option value="${d.id}" ${ref && Number(ref.department_id)===Number(d.id)?'selected':''}>${escapeHtml(d.name)}</option>`).join('');
    fBreak.value = typeof ref?.break_minutes==='number' ? ref.break_minutes : defaultBreak(date,start,end);

    selectedUserIds = new Set((ref?.user_ids || preset?.userIds || []).map(Number));
    syncChips(); renderUserList('');
  }
  drawerClose.addEventListener('click', ()=> drawer.setAttribute('aria-hidden','true'));
  deleteShiftBtn.addEventListener('click', ()=>{ if (editingIndex>=0){ shifts.splice(editingIndex,1); drawer.setAttribute('aria-hidden','true'); runValidateAndRender(); } });
  autoBreak.addEventListener('click', ()=>{ if(!fDate.value||!fStart.value||!fEnd.value) return; fBreak.value = defaultBreak(fDate.value,fStart.value,fEnd.value); });

  // Form submit
  $('#shiftForm').addEventListener('submit', (e)=>{
    e.preventDefault(); drawerNote.hidden=true;
    const dept = Number(fDept.value); const date=fDate.value; const start=fStart.value; const end=fEnd.value;
    const title = fTitle.value.trim(); const breakMin = Math.max(0, parseInt(fBreak.value||'0',10));
    const user_ids = Array.from(selectedUserIds);
    const errs=[];
    if(!dept) errs.push('Kies afdeling'); if(!date) errs.push('Kies datum');
    if(!start||!end) errs.push('Start/Eind verplicht'); if(start&&end&&end<=start) errs.push('Eindtijd ≤ start');
    if(!user_ids.length) errs.push('Kies minimaal één medewerker');

    // ✅ client-check: per user max 1 dienst per dag
    user_ids.forEach(uid=>{
      const clash = shifts.some((s, idx)=> idx!==editingIndex && s.user_ids?.includes(uid) && s.start_at.slice(0,10)===date);
      if (clash) errs.push(`Medewerker #${uid} heeft al een dienst op ${date}`);
    });

    if(errs.length){ show(drawerNote, errs.join(' • ')); return; }

    const obj={ department_id:dept, title: title||null, start_at: parseLocal(date,start), end_at: parseLocal(date,end), break_minutes: breakMin, user_ids };
    if (editingIndex>=0) shifts[editingIndex]=obj; else shifts.push(obj);
    drawer.setAttribute('aria-hidden','true');
    runValidateAndRender();
  });

  // Validate & conflicts
  function showConflicts(list){
    if (!list || !list.length){ confPane.hidden=true; confList.innerHTML=''; confCount.textContent='0'; return; }
    confPane.hidden=false; confCount.textContent=String(list.length);
    confList.innerHTML = list.map(c=>{
      const label = c.type==='overlap' ? 'Overlap' : (c.type==='multi' ? 'Meerdere diensten' : 'Openingstijden');
      return `<div class="confItem">⚠️ <strong>${label}</strong> — ${escapeHtml(c.details||'')}
        ${c.user_name?(' • '+escapeHtml(c.user_name)) : ''} • ${escapeHtml(c.date)}
        ${c.start_at?(' '+escapeHtml(c.start_at)):""}${c.end_at?('–'+escapeHtml(c.end_at)):""}</div>`;
    }).join('');
  }
  async function validateServer(){
    try{
      const res = await fetch('/api/roster/validate.php', { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({ roster_id: rosterId, shifts })});
      const data = await res.json();
      if (!res.ok || !data.ok) return { conflicts:[], warnings:[] };
      return { conflicts: data.conflicts || [], warnings: data.warnings || [] };
    }catch(_){ return { conflicts:[], warnings:[] }; }
  }
  async function runValidateAndRender(){
    const { conflicts } = await validateServer();
    showConflicts(conflicts);
    drawShifts(conflicts);
  }

  // Save/Publish
  async function save(status){
    note.hidden=true;
    if (!rName.value.trim()) return show(note,'Geef het rooster een naam.');
    if (!weekStart.value)   return show(note,'Kies een week (maandag).');
    if (!shifts.length)     return show(note,'Voeg minimaal één dienst toe.');

    const { conflicts } = await validateServer();
    if (conflicts.length){ show(note, `Los eerst ${conflicts.length} conflict(en) op in het paneel hieronder.`); showConflicts(conflicts); drawShifts(conflicts); return; }

    try{
      const start = weekStart.value; const end = ymd(addDays(new Date(start),6));
      const res = await fetch('/api/roster/upsert.php', { method:'POST', headers:{'Content-Type':'application/json'},
        body: JSON.stringify({ roster_id: rosterId, name:rName.value.trim(), start_date:start, end_date:end, shifts, status })
      });
      const data = await res.json();
      if (!res.ok || !data.ok) throw new Error(data?.error || 'Opslaan mislukt');
      rosterId = data.roster_id || rosterId;
      show(note, status==='published' ? 'Rooster gepubliceerd!' : 'Concept opgeslagen!', true);
    } catch(e){ show(note, e.message || 'Opslaan mislukt'); }
  }
  saveDraft.addEventListener('click', ()=> save('draft'));
  publishBtn.addEventListener('click', ()=> save('published'));

  // Week leegmaken
  clearWeekBtn.addEventListener('click', ()=>{
    if (!shifts.length) return show(note,'Geen diensten om te verwijderen.');
    const ok = confirm('Weet je zeker dat je ALLE diensten in deze week uit het canvas wil verwijderen?');
    if (!ok) return;
    shifts = [];
    runValidateAndRender();
    show(note, 'Alle diensten voor deze week zijn uit het canvas verwijderd.', true);
  });

  // Auto-planner (zelfde als eerder – UI)
  function renderAutoMatrix(){
    const labels=['Ma','Di','Wo','Do','Vr','Za','Zo']; const ids=Array.from(selectedDeptIds);
    if (!ids.length){ autoMatrix.innerHTML = '<p class="muted">Kies eerst afdelingen.</p>'; return; }
    const table=document.createElement('table'); table.className='table';
    const thead=document.createElement('thead'); const trh=document.createElement('tr');
    trh.innerHTML = `<th>Afdeling</th>${labels.map(l=>`<th>${l}</th>`).join('')}`; thead.appendChild(trh); table.appendChild(thead);
    const tbody=document.createElement('tbody');
    ids.forEach(id=>{
      const dep=departments.find(d=>Number(d.id)===Number(id));
      const tr=document.createElement('tr'); tr.dataset.dept=id;
      let cells=`<td>${escapeHtml(dep?.name||('Dept '+id))}</td>`;
      for(let w=1;w<=7;w++) cells += `<td><input type="number" min="0" value="1" style="width:70px" data-w="${w}"></td>`;
      tr.innerHTML=cells; tbody.appendChild(tr);
    });
    table.appendChild(tbody); autoMatrix.innerHTML=''; autoMatrix.appendChild(table);
  }
  autoBtn.addEventListener('click', ()=>{ renderAutoMatrix(); autoModal.setAttribute('aria-hidden','false'); autoNote.hidden=true; autoPreview.hidden=true; concept=null; applyAppend.disabled=true; applyReplace.disabled=true; });
  autoClose.addEventListener('click', ()=> autoModal.setAttribute('aria-hidden','true'));
  genConcept.addEventListener('click', async ()=>{
    autoNote.hidden=true; autoPreview.hidden = true; concept=null;
    if (!weekStart.value) return show(autoNote,'Kies eerst een week (maandag).');
    const ids=Array.from(selectedDeptIds); if (!ids.length) return show(autoNote,'Kies minimaal één afdeling.');
    const staffing={}; autoMatrix.querySelectorAll('tbody tr').forEach(tr=>{ const dept=Number(tr.dataset.dept); staffing[dept]={}; tr.querySelectorAll('input[type="number"]').forEach(inp=>{ const w=Number(inp.dataset.w); staffing[dept][w]=Math.max(0,parseInt(inp.value||'0',10)); }); });
    genConcept.disabled=true; genConcept.innerHTML='<span class="spinner"></span> Genereren…';
    try{
      const res = await fetch('/api/roster/auto-generate.php', { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({ week_start: weekStart.value, departments: ids, staffing, max_shift_hours: 8, use_business_hours: true })});
      const data = await res.json(); if (!res.ok || !data.ok) throw new Error(data?.error||'Genereren mislukt');
      concept = data.shifts || [];
      autoPreview.innerHTML = summarizeConcept(concept); autoPreview.hidden=false;
      applyAppend.disabled=false; applyReplace.disabled=false;
      show(autoNote, data.warnings?.length ? `Klaar met waarschuwingen (${data.warnings.length}).` : 'Concept gereed!', !data.warnings?.length);
    }catch(e){ show(autoNote, e.message || 'Genereren mislukt'); }
    finally{ genConcept.disabled=false; genConcept.textContent='Genereer concept'; }
  });
  function summarizeConcept(list){
    const byDayDept={}; list.forEach(s=>{ const d=s.start_at.slice(0,10); const dep=Number(s.department_id); byDayDept[d]??={}; byDayDept[d][dep]??=0; byDayDept[d][dep]+=(s.user_ids?.length||0); });
    const days=Object.keys(byDayDept).sort(); if(!days.length) return '<p class="muted">Geen voorstellen.</p>';
    let html='<ul>'; days.forEach(d=>{ html += `<li><strong>${new Date(d+'T00:00:00').toLocaleDateString('nl-NL',{weekday:'long', day:'2-digit', month:'2-digit'})}</strong>: `; const parts=Object.entries(byDayDept[d]).map(([dep,cnt])=>`${escapeHtml(departments.find(x=>Number(x.id)===Number(dep))?.name || ('Dept '+dep))}: ${cnt}`); html += parts.join(' • ') + '</li>'; }); html+='</ul>'; return html;
  }
  function dedupeAppend(base, add){ const key=s=>`${s.department_id}|${s.start_at}|${s.end_at}|${(s.user_ids||[]).slice().sort((a,b)=>a-b).join(',')}`; const seen=new Set(base.map(key)); add.forEach(s=>{ const k=key(s); if(!seen.has(k)){ base.push(s); seen.add(k);} }); return base; }
  applyAppend.addEventListener('click', ()=>{ if(!concept) return; shifts = dedupeAppend(shifts, concept); autoModal.setAttribute('aria-hidden','true'); runValidateAndRender(); });
  applyReplace.addEventListener('click', ()=>{ if(!concept) return; shifts = concept.slice(); autoModal.setAttribute('aria-hidden','true'); runValidateAndRender(); });

  // Help
  helpBtn.addEventListener('click', ()=> document.getElementById('helpModal').setAttribute('aria-hidden','false'));
  document.getElementById('helpClose').addEventListener('click', ()=> document.getElementById('helpModal').setAttribute('aria-hidden','true'));

  // Week controls
  function setMondayInput(d){ weekStart.value = ymd(d); }
  prev.addEventListener('click', ()=>{ weekMonday = addDays(weekMonday,-7); setMondayInput(weekMonday); renderBoard(); runValidateAndRender(); });
  next.addEventListener('click', ()=>{ weekMonday = addDays(weekMonday, 7); setMondayInput(weekMonday); renderBoard(); runValidateAndRender(); });
  weekStart.addEventListener('change', ()=>{ const chosen = new Date(weekStart.value+'T00:00:00'); weekMonday = getMonday(chosen); setMondayInput(weekMonday); renderBoard(); runValidateAndRender(); });

  // Init
  await loadDepartments(); await loadUsers();
  setMondayInput(weekMonday); rName.value = `Week ${weekStart.value}`; buildLanes(); renderBoard(); runValidateAndRender();
})();
